
//
// (c) 2006-2009 devolo AG, Aachen (Germany)
//

#include <ctype.h>
#include <stdint.h>
#include <string.h>
#include <string>
#include <algorithm>
#include <functional>
#include "hptools.h"


namespace {

typedef struct
{
    uint32_t        iState[ 4 ];        /* state (ABCD) */
    uint32_t        iBits[ 2 ];         /* number of bits, modulo 2^64 (lsb first) */
    unsigned char   cInBuf[ 64 ];       /* input buffer */
} 
Md5_tContext;

void Md5_vInit(  
  Md5_tContext* pCtx)     /* ptr to context */
{
    pCtx->iState[ 0 ] = 0x67452301;
    pCtx->iState[ 1 ] = 0xefcdab89;
    pCtx->iState[ 2 ] = 0x98badcfe;
    pCtx->iState[ 3 ] = 0x10325476;

    pCtx->iBits[ 0 ] = 0;
    pCtx->iBits[ 1 ] = 0;
}

void Md5_vTransform(   
  uint32_t* pState,          /* ptr to state */
  const uint32_t* pIn )      /* ptr to working buffer */
{
    /* The four core functions - mF1 is optimized somewhat */

    /* #define mF1(x, y, z) (x & y | ~x & z) */
    #define mF1( x, y, z )           ( z ^ ( x & ( y ^ z ) ) )
    #define mF2( x, y, z )           mF1( z, x, y )
    #define mF3( x, y, z )           ( x ^ y ^ z )
    #define mF4( x, y, z )           ( y ^ ( x | ~z ) )

    /* This is the central step in the MD5 algorithm. */
    #define mStep( f, w, x, y, z, data, s )             {                                   \
                                                            w += f( x, y, z ) + data;       \
                                                            w = w << s | w>>( 32 - s );     \
                                                            w += x;                         \
                                                        }

    register uint32_t a, b, c, d;

    a = pState[0];
    b = pState[1];
    c = pState[2];
    d = pState[3];

    mStep( mF1, a, b, c, d, pIn[  0 ] + 0xd76aa478,  7 );
    mStep( mF1, d, a, b, c, pIn[  1 ] + 0xe8c7b756, 12 );
    mStep( mF1, c, d, a, b, pIn[  2 ] + 0x242070db, 17 );
    mStep( mF1, b, c, d, a, pIn[  3 ] + 0xc1bdceee, 22 );
    mStep( mF1, a, b, c, d, pIn[  4 ] + 0xf57c0faf,  7 );
    mStep( mF1, d, a, b, c, pIn[  5 ] + 0x4787c62a, 12 );
    mStep( mF1, c, d, a, b, pIn[  6 ] + 0xa8304613, 17 );
    mStep( mF1, b, c, d, a, pIn[  7 ] + 0xfd469501, 22 );
    mStep( mF1, a, b, c, d, pIn[  8 ] + 0x698098d8,  7 );
    mStep( mF1, d, a, b, c, pIn[  9 ] + 0x8b44f7af, 12 );
    mStep( mF1, c, d, a, b, pIn[ 10 ] + 0xffff5bb1, 17 );
    mStep( mF1, b, c, d, a, pIn[ 11 ] + 0x895cd7be, 22 );
    mStep( mF1, a, b, c, d, pIn[ 12 ] + 0x6b901122,  7 );
    mStep( mF1, d, a, b, c, pIn[ 13 ] + 0xfd987193, 12 );
    mStep( mF1, c, d, a, b, pIn[ 14 ] + 0xa679438e, 17 );
    mStep( mF1, b, c, d, a, pIn[ 15 ] + 0x49b40821, 22 );

    mStep( mF2, a, b, c, d, pIn[  1 ] + 0xf61e2562,  5 );
    mStep( mF2, d, a, b, c, pIn[  6 ] + 0xc040b340,  9 );
    mStep( mF2, c, d, a, b, pIn[ 11 ] + 0x265e5a51, 14 );
    mStep( mF2, b, c, d, a, pIn[  0 ] + 0xe9b6c7aa, 20 );
    mStep( mF2, a, b, c, d, pIn[  5 ] + 0xd62f105d,  5 );
    mStep( mF2, d, a, b, c, pIn[ 10 ] + 0x02441453,  9 );
    mStep( mF2, c, d, a, b, pIn[ 15 ] + 0xd8a1e681, 14 );
    mStep( mF2, b, c, d, a, pIn[  4 ] + 0xe7d3fbc8, 20 );
    mStep( mF2, a, b, c, d, pIn[  9 ] + 0x21e1cde6,  5 );
    mStep( mF2, d, a, b, c, pIn[ 14 ] + 0xc33707d6,  9 );
    mStep( mF2, c, d, a, b, pIn[  3 ] + 0xf4d50d87, 14 );
    mStep( mF2, b, c, d, a, pIn[  8 ] + 0x455a14ed, 20 );
    mStep( mF2, a, b, c, d, pIn[ 13 ] + 0xa9e3e905,  5 );
    mStep( mF2, d, a, b, c, pIn[  2 ] + 0xfcefa3f8,  9 );
    mStep( mF2, c, d, a, b, pIn[  7 ] + 0x676f02d9, 14 );
    mStep( mF2, b, c, d, a, pIn[ 12 ] + 0x8d2a4c8a, 20 );

    mStep( mF3, a, b, c, d, pIn[  5 ] + 0xfffa3942,  4 );
    mStep( mF3, d, a, b, c, pIn[  8 ] + 0x8771f681, 11 );
    mStep( mF3, c, d, a, b, pIn[ 11 ] + 0x6d9d6122, 16 );
    mStep( mF3, b, c, d, a, pIn[ 14 ] + 0xfde5380c, 23 );
    mStep( mF3, a, b, c, d, pIn[  1 ] + 0xa4beea44, 4  );
    mStep( mF3, d, a, b, c, pIn[  4 ] + 0x4bdecfa9, 11 );
    mStep( mF3, c, d, a, b, pIn[  7 ] + 0xf6bb4b60, 16 );
    mStep( mF3, b, c, d, a, pIn[ 10 ] + 0xbebfbc70, 23 );
    mStep( mF3, a, b, c, d, pIn[ 13 ] + 0x289b7ec6,  4 );
    mStep( mF3, d, a, b, c, pIn[  0 ] + 0xeaa127fa, 11 );
    mStep( mF3, c, d, a, b, pIn[  3 ] + 0xd4ef3085, 16 );
    mStep( mF3, b, c, d, a, pIn[  6 ] + 0x04881d05, 23 );
    mStep( mF3, a, b, c, d, pIn[  9 ] + 0xd9d4d039,  4 );
    mStep( mF3, d, a, b, c, pIn[ 12 ] + 0xe6db99e5, 11 );
    mStep( mF3, c, d, a, b, pIn[ 15 ] + 0x1fa27cf8, 16 );
    mStep( mF3, b, c, d, a, pIn[  2 ] + 0xc4ac5665, 23 );

    mStep( mF4, a, b, c, d, pIn[  0 ] + 0xf4292244,  6 );
    mStep( mF4, d, a, b, c, pIn[  7 ] + 0x432aff97, 10 );
    mStep( mF4, c, d, a, b, pIn[ 14 ] + 0xab9423a7, 15 );
    mStep( mF4, b, c, d, a, pIn[  5 ] + 0xfc93a039, 21 );
    mStep( mF4, a, b, c, d, pIn[ 12 ] + 0x655b59c3,  6 );
    mStep( mF4, d, a, b, c, pIn[ 3  ] + 0x8f0ccc92, 10 );
    mStep( mF4, c, d, a, b, pIn[ 10 ] + 0xffeff47d, 15 );
    mStep( mF4, b, c, d, a, pIn[  1 ] + 0x85845dd1, 21 );
    mStep( mF4, a, b, c, d, pIn[  8 ] + 0x6fa87e4f,  6 );
    mStep( mF4, d, a, b, c, pIn[ 15 ] + 0xfe2ce6e0, 10 );
    mStep( mF4, c, d, a, b, pIn[  6 ] + 0xa3014314, 15 );
    mStep( mF4, b, c, d, a, pIn[ 13 ] + 0x4e0811a1, 21 );
    mStep( mF4, a, b, c, d, pIn[  4 ] + 0xf7537e82,  6 );
    mStep( mF4, d, a, b, c, pIn[ 11 ] + 0xbd3af235, 10 );
    mStep( mF4, c, d, a, b, pIn[  2 ] + 0x2ad7d2bb, 15 );
    mStep( mF4, b, c, d, a, pIn[  9 ] + 0xeb86d391, 21 );

    pState[ 0 ] += a;
    pState[ 1 ] += b;
    pState[ 2 ] += c;
    pState[ 3 ] += d;
}

void Md5_vUpdate(
  Md5_tContext* pCtx,            /* ptr to context */
  const unsigned char* pBuf,     /* ptr to working buffer */
  unsigned iLen)                 /* length of buffer */
{
    uint32_t t;

    /* Update bitcount */

    t = pCtx->iBits[ 0 ];
    if ((pCtx->iBits[ 0 ] = t + ( ( uint32_t )iLen << 3 ) ) < t )
    {
        pCtx->iBits[ 1 ]++;     /* Carry from low to high */
    }
    pCtx->iBits[ 1 ] += iLen >> 29;

    t = ( t >> 3 ) & 0x3f;    /* Bytes already in shsInfo->data */

    /* Handle any leading odd-sized chunks */

    if ( t )
    {
        unsigned char *p = ( unsigned char * )pCtx->cInBuf + t;

        t = 64 - t;
        if ( iLen < t )
        {
            memcpy( p, pBuf, iLen );
            return;
        }
        memcpy( p, pBuf, t );
        Md5_vTransform( pCtx->iState, ( uint32_t * )pCtx->cInBuf );
        pBuf += t;
        iLen -= t;
    }
    /* Process data in 64-byte chunks */

    while ( iLen >= 64 )
    {
        memcpy( pCtx->cInBuf, pBuf, 64 );
        Md5_vTransform( pCtx->iState, ( uint32_t * )pCtx->cInBuf );
        pBuf += 64;
        iLen -= 64;
    }

    /* Handle any remaining bytes of data. */

    memcpy( pCtx->cInBuf, pBuf, iLen );
}

void Md5_vFinal(   
  Md5_tContext* pCtx)          /* ptr to context */
{
    unsigned        iCount;
    unsigned char   *pC;

    /* Compute number of bytes mod 64 */
    iCount = ( pCtx->iBits[ 0 ] >> 3 ) & 0x3F;

    /* Set the first char of padding to 0x80.  This is safe since there is
       always at least one byte free */
    pC = pCtx->cInBuf + iCount;
    *pC++ = 0x80;

    /* Bytes of padding needed to make 64 bytes */
    iCount = (64 - 1) - iCount;

    /* Pad out to 56 mod 64 */
    if ( iCount < 8 )
    {
        /* Two lots of padding:  Pad the first block to 64 bytes */
        memset( pC, 0, iCount );
        Md5_vTransform( pCtx->iState, ( uint32_t * ) pCtx->cInBuf );

        /* Now fill the next block with 56 bytes */
        memset( pCtx->cInBuf, 0, 56 );
    }
    else
    {
        /* Pad block to 56 bytes */
        memset( pC, 0, iCount - 8 );
    }

    /* Append length in bits and transform */
    ( ( uint32_t * )pCtx->cInBuf)[ 14 ] = pCtx->iBits[ 0 ];
    ( ( uint32_t * )pCtx->cInBuf)[ 15 ] = pCtx->iBits[ 1 ];

    Md5_vTransform( pCtx->iState, ( uint32_t * )pCtx->cInBuf );
}

/* Digests a string */
void MD5_pDigestString(
  const unsigned char* pString,
  int iLen,
  unsigned char* pDigestBuffer)
{
  Md5_tContext sContext;

  Md5_vInit(&sContext);
  Md5_vUpdate(&sContext, pString, iLen);
  Md5_vFinal(&sContext);
    
  memcpy(pDigestBuffer, sContext.iState, 16);
  memset(&sContext, 0, sizeof(sContext));        /* In case it's sensitive */
}

static const size_t SALT_SIZE = 8;
static const size_t MAX_SECRET_SIZE = 64;

const unsigned char DAKSalt[] = {0x08, 0x85, 0x6D, 0xAF, 0x7C, 0xF5, 0x81, 0x85};
const unsigned char NMKSalt[] = {0x08, 0x85, 0x6D, 0xAF, 0x7C, 0xF5, 0x81, 0x86};

#define GET_UINT32(n,b,i)                       \
{                                               \
    (n) = ( (uint32_t) (b)[(i)    ] << 24 )       \
        | ( (uint32_t) (b)[(i) + 1] << 16 )       \
        | ( (uint32_t) (b)[(i) + 2] <<  8 )       \
        | ( (uint32_t) (b)[(i) + 3]       );      \
}

#define PUT_UINT32(n,b,i)                       \
{                                               \
    (b)[(i)    ] = (unsigned char) ( (n) >> 24 );       \
    (b)[(i) + 1] = (unsigned char) ( (n) >> 16 );       \
    (b)[(i) + 2] = (unsigned char) ( (n) >>  8 );       \
    (b)[(i) + 3] = (unsigned char) ( (n)       );       \
}

class CSHA256Engine {
public:
  CSHA256Engine(void);
  const unsigned char* HashNMK(const unsigned char* iSecret);
  const unsigned char* HashDAK(const unsigned char* iSecret);
  const unsigned char* HashNID(const unsigned char* iSecret);
  ~CSHA256Engine(void);
private:
  struct sha256_context {
    sha256_context() 
    {
      memset(total,0,sizeof(total));
      memset(state,0,sizeof(state));
      memset(buffer,0,sizeof(buffer));
    }
    inline void Initialize() {
      // Clear the Internal buffers...
      memset(this, 0, sizeof(sha256_context));
      // Set the Initial Hash Value (per FIPS180-2 sec 5.3.2)
        state[0] = 0x6A09E667;
        state[1] = 0xBB67AE85;
        state[2] = 0x3C6EF372;
        state[3] = 0xA54FF53A;
        state[4] = 0x510E527F;
        state[5] = 0x9B05688C;
        state[6] = 0x1F83D9AB;
        state[7] = 0x5BE0CD19;
    }
    uint32_t total[2];
    uint32_t state[8];
    unsigned char buffer[64];
  };
  class CSaltedSecret {
  public:
    // CTOR: create a salted secret in this format -> SECRET+SALT
    CSaltedSecret(const unsigned char* iSecret, const unsigned char* iSalt = NULL)
      : m_Length  (0)
    {
      unsigned char l = 0;
      memset(m_Value, 0x00, sizeof(m_Value));
      // Make sure the Secret is Plausable...
      if(iSecret) {
        // Secret is a NULL ternminated string; If the Secret exceeds the MAX secret size, truncate
        // to MAX secret size...
        if((l = (unsigned char) strlen((char*)iSecret)) > MAX_SECRET_SIZE)
          l = MAX_SECRET_SIZE;
        memcpy(m_Value, (char*)iSecret, l);
      }
      if(!iSalt)//must be for Hashing NID from NMK - size is 16
      { l = 16;}
      m_Length = (unsigned char) (m_Length + l);
      // SALT is optional, but if used, it must be 8 octets long...
      if(iSalt) {
        // Post-pend the SALT to the SECRET...
        memcpy(&m_Value[m_Length], (char*)iSalt, SALT_SIZE);
        m_Length += SALT_SIZE;
      }
    }
    unsigned char getLength() const { return m_Length; }
    unsigned char* getValue() { return m_Value; } 
  private:
    unsigned char      m_Length;
    unsigned char      m_Value[MAX_SECRET_SIZE + SALT_SIZE];
  };
  class CSaltedNMK : public CSaltedSecret {
  public:
    CSaltedNMK(const unsigned char* iSecret)
      : CSaltedSecret(iSecret, NMKSalt)
    {
    }
  };
  class CSaltedDAK : public CSaltedSecret {
  public:
    CSaltedDAK(const unsigned char* iSecret)
      : CSaltedSecret(iSecret, DAKSalt)
    {
    }
  };
  class CSaltedNID : public CSaltedSecret {
  public:
    CSaltedNID(const unsigned char* iSecret)
      : CSaltedSecret(iSecret)
    {
    }
  };
  //
  void Process(const unsigned char data[64]);
  void Update(const unsigned char* input, uint32_t length);
  void Finish();
  //
  unsigned char  m_HashValue[32];
  sha256_context m_ctx;
};


CSHA256Engine::CSHA256Engine(void)
{
  memset(m_HashValue,0,sizeof(m_HashValue));
}

CSHA256Engine::~CSHA256Engine(void)
{
}

const unsigned char* CSHA256Engine::HashNID( const unsigned char* iSecret)
{
  // Set the Initial State for Hashing...
  m_ctx.Initialize();
  memset(m_HashValue, 0x00, sizeof(m_HashValue));
  // add Salt to the Key...
  CSaltedNID  sk(iSecret);
  // Per HomePlug Spec run the Hash 5 times; Once on the SaltedSecret and then on the Hash itself ...
  Update(sk.getValue(), sk.getLength());
  Finish();
  for(int j = 0; j < 4; j++) {
    m_ctx.Initialize();
    Update(m_HashValue, 32);
    Finish();
  }
  // Return a const pointer to the internal 32 Byte Hash buffer. Implementators are free to use fewer Bytes
  // but the full 32 Byte Hash is valid. Also, this also eliminates questions regarding validity of the output
  //buffer.
  return m_HashValue;
}

const unsigned char* CSHA256Engine::HashDAK(const unsigned char* iSecret)
{
  // Set the Initial State for Hashing...
  m_ctx.Initialize();
  memset(m_HashValue, 0x00, sizeof(m_HashValue));
  // add Salt to the Key...
  CSaltedDAK  sk(iSecret);
  // Per HomePlug Spec run the Hash 1000 times; Once on the SaltedSecret and then on the Hash itself ...
  Update(sk.getValue(), sk.getLength());
  Finish();
  for(int j = 0; j < 999; j++) {
    m_ctx.Initialize();
    Update(m_HashValue, 32);
    Finish();
  }
  // Return a const pointer to the internal 32 Byte Hash buffer. Implementators are free to use fewer Bytes
  // but the full 32 Byte Hash is valid. Also, this also eliminates questions regarding validity of the output
  //buffer.
  return m_HashValue;
}

const unsigned char* CSHA256Engine::HashNMK(const unsigned char* iSecret)
{
  // Set the Initial State for Hashing...
  m_ctx.Initialize();
  memset(m_HashValue, 0x00, sizeof(m_HashValue));
  // add Salt to the Key...
  CSaltedNMK  sk(iSecret);
  // Per HomePlug Spec run the Hash 1000 times; Once on the SaltedSecret and then on the Hash itself ...
  Update(sk.getValue(), sk.getLength());
  Finish();
  for(int j = 0; j < 999; j++) {
    m_ctx.Initialize();
    Update(m_HashValue, 32);
    Finish();
  }
  // Return a const pointer to the internal 32 Byte Hash buffer. Implementators are free to use fewer Bytes
  // but the full 32 Byte Hash is valid. Also, this also eliminates questions regarding validity of the output
  //buffer.
  return m_HashValue;
}

void CSHA256Engine::Process(const unsigned char data[64])
{
  uint32_t temp1, temp2, W[64];
  uint32_t A, B, C, D, E, F, G, H;
  //
  GET_UINT32( W[0],  data,  0 );
  GET_UINT32( W[1],  data,  4 );
  GET_UINT32( W[2],  data,  8 );
  GET_UINT32( W[3],  data, 12 );
  GET_UINT32( W[4],  data, 16 );
  GET_UINT32( W[5],  data, 20 );
  GET_UINT32( W[6],  data, 24 );
  GET_UINT32( W[7],  data, 28 );
  GET_UINT32( W[8],  data, 32 );
  GET_UINT32( W[9],  data, 36 );
  GET_UINT32( W[10], data, 40 );
  GET_UINT32( W[11], data, 44 );
  GET_UINT32( W[12], data, 48 );
  GET_UINT32( W[13], data, 52 );
  GET_UINT32( W[14], data, 56 );
  GET_UINT32( W[15], data, 60 );

#define  SHR(x,n) ((x & 0xFFFFFFFF) >> n)
#define ROTR(x,n) (SHR(x,n) | (x << (32 - n)))

#define S0(x) (ROTR(x, 7) ^ ROTR(x,18) ^  SHR(x, 3))
#define S1(x) (ROTR(x,17) ^ ROTR(x,19) ^  SHR(x,10))

#define S2(x) (ROTR(x, 2) ^ ROTR(x,13) ^ ROTR(x,22))
#define S3(x) (ROTR(x, 6) ^ ROTR(x,11) ^ ROTR(x,25))

#define F0(x,y,z) ((x & y) | (z & (x | y)))
#define F1(x,y,z) (z ^ (x & (y ^ z)))

#define R(t)                                    \
(                                               \
    W[t] = S1(W[t -  2]) + W[t -  7] +          \
           S0(W[t - 15]) + W[t - 16]            \
)

#define P(a,b,c,d,e,f,g,h,x,K)                  \
{                                               \
    temp1 = h + S3(e) + F1(e,f,g) + K + x;      \
    temp2 = S2(a) + F0(a,b,c);                  \
    d += temp1; h = temp1 + temp2;              \
}

    A = m_ctx.state[0];
    B = m_ctx.state[1];
    C = m_ctx.state[2];
    D = m_ctx.state[3];
    E = m_ctx.state[4];
    F = m_ctx.state[5];
    G = m_ctx.state[6];
    H = m_ctx.state[7];

    P( A, B, C, D, E, F, G, H, W[ 0], 0x428A2F98 );
    P( H, A, B, C, D, E, F, G, W[ 1], 0x71374491 );
    P( G, H, A, B, C, D, E, F, W[ 2], 0xB5C0FBCF );
    P( F, G, H, A, B, C, D, E, W[ 3], 0xE9B5DBA5 );
    P( E, F, G, H, A, B, C, D, W[ 4], 0x3956C25B );
    P( D, E, F, G, H, A, B, C, W[ 5], 0x59F111F1 );
    P( C, D, E, F, G, H, A, B, W[ 6], 0x923F82A4 );
    P( B, C, D, E, F, G, H, A, W[ 7], 0xAB1C5ED5 );
    P( A, B, C, D, E, F, G, H, W[ 8], 0xD807AA98 );
    P( H, A, B, C, D, E, F, G, W[ 9], 0x12835B01 );
    P( G, H, A, B, C, D, E, F, W[10], 0x243185BE );
    P( F, G, H, A, B, C, D, E, W[11], 0x550C7DC3 );
    P( E, F, G, H, A, B, C, D, W[12], 0x72BE5D74 );
    P( D, E, F, G, H, A, B, C, W[13], 0x80DEB1FE );
    P( C, D, E, F, G, H, A, B, W[14], 0x9BDC06A7 );
    P( B, C, D, E, F, G, H, A, W[15], 0xC19BF174 );
    P( A, B, C, D, E, F, G, H, R(16), 0xE49B69C1 );
    P( H, A, B, C, D, E, F, G, R(17), 0xEFBE4786 );
    P( G, H, A, B, C, D, E, F, R(18), 0x0FC19DC6 );
    P( F, G, H, A, B, C, D, E, R(19), 0x240CA1CC );
    P( E, F, G, H, A, B, C, D, R(20), 0x2DE92C6F );
    P( D, E, F, G, H, A, B, C, R(21), 0x4A7484AA );
    P( C, D, E, F, G, H, A, B, R(22), 0x5CB0A9DC );
    P( B, C, D, E, F, G, H, A, R(23), 0x76F988DA );
    P( A, B, C, D, E, F, G, H, R(24), 0x983E5152 );
    P( H, A, B, C, D, E, F, G, R(25), 0xA831C66D );
    P( G, H, A, B, C, D, E, F, R(26), 0xB00327C8 );
    P( F, G, H, A, B, C, D, E, R(27), 0xBF597FC7 );
    P( E, F, G, H, A, B, C, D, R(28), 0xC6E00BF3 );
    P( D, E, F, G, H, A, B, C, R(29), 0xD5A79147 );
    P( C, D, E, F, G, H, A, B, R(30), 0x06CA6351 );
    P( B, C, D, E, F, G, H, A, R(31), 0x14292967 );
    P( A, B, C, D, E, F, G, H, R(32), 0x27B70A85 );
    P( H, A, B, C, D, E, F, G, R(33), 0x2E1B2138 );
    P( G, H, A, B, C, D, E, F, R(34), 0x4D2C6DFC );
    P( F, G, H, A, B, C, D, E, R(35), 0x53380D13 );
    P( E, F, G, H, A, B, C, D, R(36), 0x650A7354 );
    P( D, E, F, G, H, A, B, C, R(37), 0x766A0ABB );
    P( C, D, E, F, G, H, A, B, R(38), 0x81C2C92E );
    P( B, C, D, E, F, G, H, A, R(39), 0x92722C85 );
    P( A, B, C, D, E, F, G, H, R(40), 0xA2BFE8A1 );
    P( H, A, B, C, D, E, F, G, R(41), 0xA81A664B );
    P( G, H, A, B, C, D, E, F, R(42), 0xC24B8B70 );
    P( F, G, H, A, B, C, D, E, R(43), 0xC76C51A3 );
    P( E, F, G, H, A, B, C, D, R(44), 0xD192E819 );
    P( D, E, F, G, H, A, B, C, R(45), 0xD6990624 );
    P( C, D, E, F, G, H, A, B, R(46), 0xF40E3585 );
    P( B, C, D, E, F, G, H, A, R(47), 0x106AA070 );
    P( A, B, C, D, E, F, G, H, R(48), 0x19A4C116 );
    P( H, A, B, C, D, E, F, G, R(49), 0x1E376C08 );
    P( G, H, A, B, C, D, E, F, R(50), 0x2748774C );
    P( F, G, H, A, B, C, D, E, R(51), 0x34B0BCB5 );
    P( E, F, G, H, A, B, C, D, R(52), 0x391C0CB3 );
    P( D, E, F, G, H, A, B, C, R(53), 0x4ED8AA4A );
    P( C, D, E, F, G, H, A, B, R(54), 0x5B9CCA4F );
    P( B, C, D, E, F, G, H, A, R(55), 0x682E6FF3 );
    P( A, B, C, D, E, F, G, H, R(56), 0x748F82EE );
    P( H, A, B, C, D, E, F, G, R(57), 0x78A5636F );
    P( G, H, A, B, C, D, E, F, R(58), 0x84C87814 );
    P( F, G, H, A, B, C, D, E, R(59), 0x8CC70208 );
    P( E, F, G, H, A, B, C, D, R(60), 0x90BEFFFA );
    P( D, E, F, G, H, A, B, C, R(61), 0xA4506CEB );
    P( C, D, E, F, G, H, A, B, R(62), 0xBEF9A3F7 );
    P( B, C, D, E, F, G, H, A, R(63), 0xC67178F2 );

    m_ctx.state[0] += A;
    m_ctx.state[1] += B;
    m_ctx.state[2] += C;
    m_ctx.state[3] += D;
    m_ctx.state[4] += E;
    m_ctx.state[5] += F;
    m_ctx.state[6] += G;
    m_ctx.state[7] += H;
}

void CSHA256Engine::Update(const unsigned char *input, uint32_t length)
{
  uint32_t left, fill;
  //
    if(!length)
      return;
    left = m_ctx.total[0] & 0x3F;
    fill = 64 - left;
    m_ctx.total[0] += length;
    m_ctx.total[0] &= 0xFFFFFFFF;
    if(m_ctx.total[0] < length)
        m_ctx.total[1]++;
    if(left && length >= fill)
    {
        memcpy((void*)(m_ctx.buffer + left), (void*)input, (uint16_t) fill);
        Process(m_ctx.buffer);
        length -= fill;
        input  += fill;
        left = 0;
    }
  while(length >= 64)
  {
    Process( input );
    length -= 64;
    input  += 64;
  }
  if(length) {
    memcpy((void*)(m_ctx.buffer + left), (void*)input, (uint16_t) length);
  }
}

static unsigned char sha256_padding[64] =
{
 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

void CSHA256Engine::Finish()
{
  // Locals
  uint32_t last, padn;
  uint32_t high, low;
  unsigned char msglen[8];
  //
  high = ( m_ctx.total[0] >> 29 ) | ( m_ctx.total[1] <<  3 );
  low  = ( m_ctx.total[0] <<  3 );

  PUT_UINT32( high, msglen, 0 );
  PUT_UINT32( low,  msglen, 4 );

  last = m_ctx.total[0] & 0x3F;
  padn = ( last < 56 ) ? ( 56 - last ) : ( 120 - last );

  Update(sha256_padding, padn);
  Update( msglen, 8 );

  PUT_UINT32( m_ctx.state[0], m_HashValue,  0 );
  PUT_UINT32( m_ctx.state[1], m_HashValue,  4 );
  PUT_UINT32( m_ctx.state[2], m_HashValue,  8 );
  PUT_UINT32( m_ctx.state[3], m_HashValue, 12 );
  PUT_UINT32( m_ctx.state[4], m_HashValue, 16 );
  PUT_UINT32( m_ctx.state[5], m_HashValue, 20 );
  PUT_UINT32( m_ctx.state[6], m_HashValue, 24 );
  PUT_UINT32( m_ctx.state[7], m_HashValue, 28 );
}

} // namespace

void HomePlugTools::KeyFromPassword(
  const std::string& strPassword,
  unsigned char cKEY[8])
{
  std::string strSaltedPassword = strPassword + std::string("\x08\x85\x6D\xAF\x7C\xF5\x81\x85", 8);
  unsigned char cDigestBuffer[16];
  MD5_pDigestString((const unsigned char*)strSaltedPassword.data(), strSaltedPassword.size(), cDigestBuffer);

  for(int i = 2; i <= 1000; i++)
    MD5_pDigestString(cDigestBuffer, 16, cDigestBuffer);

  memcpy(cKEY, cDigestBuffer, 8);

  for(int i = 0; i < 8; i++)
  {
    int iOneBits = 0;

    for(int j = 0x02; j != 0x00; j <<= 1)
      if(j & cKEY[i])
        iOneBits++;

    if((iOneBits & 0x01))
      cKEY[i] &= 0xFE;
    else
      cKEY[i] |= 0x01;
  }
}

void HomePlugAvTools::NetworkKeyFromPassword(
  const std::string& strPassword,
  unsigned char cKEY[16])
{
  unsigned char cPassword[MAX_SECRET_SIZE+1] = { 0 };
  strncpy((char*)cPassword, strPassword.c_str(), MAX_SECRET_SIZE);
  CSHA256Engine sha256engine;
  const unsigned char* pPasswordHash = sha256engine.HashNMK(cPassword);
  memcpy(cKEY, pPasswordHash, 16);
}

void HomePlugAvTools::DeviceAccessKeyFromSecurityID(
  const std::string& strSecurityID,
  unsigned char cDAK[16])
{
  unsigned char cSecurityID[MAX_SECRET_SIZE+1] = { 0 };
  strncpy((char*)cSecurityID, strSecurityID.c_str(), MAX_SECRET_SIZE);
  CSHA256Engine sha256engine;
  const unsigned char* pSecurityIdHash = sha256engine.HashDAK(cSecurityID);
  memcpy(cDAK, pSecurityIdHash, 16);
}

